<?php
declare(strict_types=1);

/**
 * Export CSV des votes d’un sondage
 * GET params:
 *   poll_id     (int)  requis
 *   option_id   (int)  optionnel (filtre option)
 *   encoding    (str)  'utf8' (défaut) ou 'excel' (UTF-8 + BOM + ligne sep=)
 *   sep         (str)  séparateur, ';' par défaut (autorise ';' ',' ou tab)
 *   decimals    (int)  décimales pour % (défaut 1)
 */

require_once dirname(__DIR__, 3) . '/mainfile.php';

xoops_loadLanguage('admin', 'sondages_pro');

// ---- Sécurité admin module (robuste) ----
$xoopsUser = $GLOBALS['xoopsUser'] ?? null;

$mid = 0;
if (isset($GLOBALS['xoopsModule']) && is_object($GLOBALS['xoopsModule'])
    && $GLOBALS['xoopsModule']->getVar('dirname') === 'sondages_pro') {
    $mid = (int)$GLOBALS['xoopsModule']->getVar('mid');
} else {
    $mHandler = xoops_getHandler('module');
    $mod = $mHandler ? $mHandler->getByDirname('sondages_pro') : null;
    if ($mod) {
        $mid = (int)$mod->getVar('mid');
    }
}
if (!is_object($xoopsUser) || $mid <= 0 || !$xoopsUser->isAdmin($mid)) {
    redirect_header(XOOPS_URL . '/modules/sondages_pro/admin/votes.php', 3, defined('_NOPERM') ? _NOPERM : 'Accès refusé.');
    exit;
}

// ---- Helpers CSV ----
/** Préfixe anti-injection Excel (=, +, -, @) sur champs texte */
function sp_csv_guard(string $s): string {
    if ($s !== '' && strpbrk($s[0], '=+-@') !== false) return "'".$s;
    return $s;
}
/** fputcsv avec garde et séparateur choisi */
function sp_putcsv($handle, array $fields, string $sep): void {
    $safe = [];
    foreach ($fields as $v) {
        $safe[] = sp_csv_guard((string)$v);
    }
    @fputcsv($handle, $safe, $sep);
}

// ---- Paramètres ----
$pollId   = (int)($_GET['poll_id'] ?? 0);
$optionId = (int)($_GET['option_id'] ?? 0);
$encoding = strtolower((string)($_GET['encoding'] ?? 'utf8'));   // utf8 | excel
$sep      = (string)($_GET['sep'] ?? ';');                        // ; , \t
$decimals = max(0, (int)($_GET['decimals'] ?? 1));

if (!in_array($sep, [';', ',', "\t"], true)) {
    $sep = ';';
}
if ($pollId <= 0) {
    redirect_header('votes.php', 2, defined('_AM_SYSTEM_DBERROR') ? _AM_SYSTEM_DBERROR : 'ID de sondage invalide.');
    exit;
}

// ---- Handlers ----
$pollHandler = xoops_getModuleHandler('poll', 'sondages_pro');
$optHandler  = xoops_getModuleHandler('option', 'sondages_pro');
$voteHandler = xoops_getModuleHandler('vote', 'sondages_pro');

if (!$pollHandler || !$optHandler || !$voteHandler) {
    redirect_header('votes.php', 2, defined('_AM_SYSTEM_DBERROR') ? _AM_SYSTEM_DBERROR : 'Handlers indisponibles.');
    exit;
}

$poll = $pollHandler->get($pollId);
if (!$poll) {
    redirect_header('votes.php', 2, defined('_AM_SYSTEM_DBERROR') ? _AM_SYSTEM_DBERROR : 'Sondage introuvable.');
    exit;
}

$xoopsDB = $GLOBALS['xoopsDB'];

// ---- Métadonnées sondage ----
// Utilise 'n' (raw) pour éviter toute entité HTML & problèmes d’accents
$question = (string)$poll->getVar('question', 'n');
$starts   = (string)$poll->getVar('starts_at', 'n');
$ends     = (string)$poll->getVar('ends_at', 'n');

// ---- Tables ----
$tblV = $xoopsDB->prefix('sp_votes');
$tblO = $xoopsDB->prefix('sp_options');
$tblU = $xoopsDB->prefix('users');

// ---- Total votes (non filtré par option) ----
$sqlTotal = "SELECT COUNT(*) FROM {$tblV} WHERE poll_id=".(int)$pollId;
list($totalVotes) = $xoopsDB->fetchRow($xoopsDB->queryF($sqlTotal));
$totalVotes = (int)$totalVotes;

// ---- Votes par option (pour % globaux) ----
$sqlOptCounts = "
  SELECT option_id, COUNT(*) AS cnt
  FROM {$tblV}
  WHERE poll_id=".(int)$pollId."
  GROUP BY option_id
";
$res = $xoopsDB->queryF($sqlOptCounts);
$countsByOption = [];
if ($res) {
    while ($row = $xoopsDB->fetchArray($res)) {
        $countsByOption[(int)$row['option_id']] = (int)$row['cnt'];
    }
}

// ---- Libellés des options (raw) ----
$opts = method_exists($optHandler,'getByPoll')
    ? $optHandler->getByPoll($pollId)
    : $optHandler->getObjects(new Criteria('poll_id', $pollId));
$labelsByOption = [];
foreach ($opts as $o) {
    $labelsByOption[(int)$o->getVar('option_id')] = (string)$o->getVar('label','n');
}

// ---- Détail des votes (filtré si option_id) ----
// LEFT JOIN sur options pour conserver d’éventuels votes orphelins (options supprimées)
$where = "v.poll_id=".(int)$pollId;
if ($optionId > 0) {
    $where .= " AND v.option_id=".(int)$optionId;
}
$sqlDetail = "
SELECT v.vote_id, v.poll_id, v.option_id, v.uid, v.ip, v.created_at,
       o.label AS option_label,
       u.uname AS uname
FROM {$tblV} v
LEFT JOIN {$tblO} o ON o.option_id = v.option_id
LEFT JOIN {$tblU} u ON u.uid = v.uid
WHERE {$where}
ORDER BY v.created_at DESC, v.vote_id DESC
";
$res = $xoopsDB->queryF($sqlDetail);
$rows = [];
$firstDt = null; $lastDt = null;

if ($res) {
    while ($row = $xoopsDB->fetchArray($res)) {
        // Normalise libellé option (raw) + fallback si orpheline
        if (!isset($row['option_label']) || $row['option_label'] === null) {
            $row['option_label'] = '#'.(int)$row['option_id'].' (option supprimée)';
        }
        $rows[] = $row;

        $dt = (string)$row['created_at'];
        if ($dt !== '') {
            if ($firstDt === null || $dt < $firstDt) $firstDt = $dt;
            if ($lastDt  === null || $dt > $lastDt)  $lastDt  = $dt;
        }
    }
}

// ---- Pré-calcul par répondant ----
$byRespondent = []; // key => ['count'=>n, 'first'=>dt]
foreach ($rows as $r) {
    $uid = (int)$r['uid'];
    $ip  = (string)$r['ip'];
    $key = ($uid > 0) ? ('U#'.$uid) : ('G#'.$ip);
    if (!isset($byRespondent[$key])) {
        $byRespondent[$key] = ['count'=>0, 'first'=>null];
    }
    $byRespondent[$key]['count']++;
    $dt = (string)$r['created_at'];
    if ($dt !== '') {
        if ($byRespondent[$key]['first'] === null || $dt < $byRespondent[$key]['first']) {
            $byRespondent[$key]['first'] = $dt;
        }
    }
}

// ---- Sortie CSV ----
$filename = sprintf('votes_poll_%d_%s.csv', $pollId, date('Ymd_His'));
header('Content-Type: text/csv; charset=UTF-8');
header('Content-Disposition: attachment; filename="'.$filename.'"');

$out = fopen('php://output', 'w');

// BOM + ligne sep= pour Excel si demandé
if ($encoding === 'excel') {
    echo "\xEF\xBB\xBF";
    echo "sep={$sep}\r\n";
}

$put = function(array $fields) use ($out, $sep) {
    sp_putcsv($out, $fields, $sep);
};

// En-tête méta
$put(["Sondage", "#{$pollId}", $question]);
$put(["Exporté le", date('Y-m-d H:i:s')]);
$put(["Période de vote", ($firstDt ?: $starts) . " → " . ($lastDt ?: $ends)]);
$put(["Total votes", (string)$totalVotes]);
$put([""]);

// Résumé par option
$put(["RÉSUMÉ PAR OPTION"]);
$put(["option_id","option_label","votes","part"]);
foreach ($labelsByOption as $oid => $label) {
    $cnt  = $countsByOption[$oid] ?? 0;
    $part = ($totalVotes > 0) ? round($cnt * 100.0 / $totalVotes, $decimals) : 0.0;
    $partStr = str_replace('.', ',', (string)$part) . '%';
    $put([(string)$oid, $label, (string)$cnt, $partStr]);
}
$put([""]);

// Détail
$put(["DÉTAIL DES VOTES"]);
$put([
  "vote_id","poll_id","option_id","option_label","part_option",
  "uid","uname","type_utilisateur","url_profil","ip",
  "votes_du_répondant","premier_vote(oui/non)","created_at"
]);

foreach ($rows as $r) {
    $oid    = (int)$r['option_id'];
    $uid    = (int)$r['uid'];
    $ip     = (string)$r['ip'];
    $key    = ($uid > 0) ? ('U#'.$uid) : ('G#'.$ip);

    $cntOpt = $countsByOption[$oid] ?? 0;
    $part   = ($totalVotes > 0) ? round($cntOpt * 100.0 / $totalVotes, $decimals) : 0.0;
    $partStr = str_replace('.', ',', (string)$part) . '%';

    $type  = ($uid > 0) ? 'membre' : 'invité';
    $url   = ($uid > 0) ? XOOPS_URL . '/userinfo.php?uid=' . $uid : '';

    $votesRespondant = $byRespondent[$key]['count'] ?? 1;
    $firstRespondant = $byRespondent[$key]['first'] ?? '';
    $isFirst         = ((string)$r['created_at'] === (string)$firstRespondant) ? 'oui' : 'non';

    $optionLabel = (string)$r['option_label']; // déjà raw/fallback

    $put([
        (string)$r['vote_id'],
        (string)$r['poll_id'],
        (string)$oid,
        $optionLabel,
        $partStr,
        (string)$uid,
        (string)($r['uname'] ?? ''),
        $type,
        $url,
        $ip,
        (string)$votesRespondant,
        $isFirst,
        (string)$r['created_at'],
    ]);
}

fclose($out);
exit;

