<?php
declare(strict_types=1);

require __DIR__ . '/header.php';
xoops_loadLanguage('admin', 'sondages_pro');

// --- Sécurité admin module
$xoopsUser = $GLOBALS['xoopsUser'] ?? null;
if (!is_object($xoopsUser) || !$xoopsUser->isAdmin($GLOBALS['xoopsModule']->getVar('mid'))) {
    redirect_header('poll.php', 3, defined('_NOPERM') ? _NOPERM : 'Accès refusé.');
    exit;
}

// --- Handlers + DB
$pollHandler = xoops_getModuleHandler('poll', 'sondages_pro');
$optHandler  = xoops_getModuleHandler('option', 'sondages_pro');
$voteHandler = xoops_getModuleHandler('vote', 'sondages_pro');
$xoopsDB     = $GLOBALS['xoopsDB'];

// --- Helpers
function sp_db_one_int(string $sql, int $default = 0): int {
    global $xoopsDB;
    $res = $xoopsDB->queryF($sql);
    if ($res) {
        $row = $xoopsDB->fetchRow($res);
        if ($row && isset($row[0])) return (int)$row[0];
    }
    // (optionnel) debug : error_log('[SP_DASH] SQL fail: '.$sql.' :: '.$xoopsDB->error());
    return $default;
}

if (!function_exists('sp_h')) {
    function sp_h(string $s): string {
        $cs = defined('_CHARSET') ? _CHARSET : 'UTF-8';
        return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, $cs);
    }
}

// --- Tables
$T_P = $xoopsDB->prefix('sp_polls');
$T_O = $xoopsDB->prefix('sp_options');
$T_V = $xoopsDB->prefix('sp_votes');
$T_U = $xoopsDB->prefix('users');

// ===========================
//  1) compteurs clés
// ===========================
$totalPolls = (int)$pollHandler->getCount(new Criteria('poll_id', 0, '>'));
$totalOpts  = (int)$optHandler->getCount(new Criteria('option_id', 0, '>'));
$totalVotes = (int)$voteHandler->getCount(new Criteria('vote_id', 0, '>'));

// Statuts
$byStatus = ['draft'=>0,'published'=>0,'closed'=>0];
$res = $xoopsDB->queryF("SELECT status, COUNT(*) c FROM {$T_P} GROUP BY status");
if ($res) {
    while ($r = $xoopsDB->fetchArray($res)) {
        $st = (string)$r['status'];
        $byStatus[$st] = (int)$r['c'];
    }
}

// Ouverts « maintenant » (logique proche sp_isPollOpen)
$sqlOpen = "
  SELECT COUNT(*)
  FROM {$T_P}
  WHERE status='published'
    AND (starts_at IS NULL OR starts_at='' OR starts_at <= NOW())
    AND (ends_at  IS NULL OR ends_at=''  OR ends_at  = '0000-00-00 00:00:00' OR ends_at >= NOW())
";
$openNow = sp_db_one_int($sqlOpen, 0);

// Activité récente
$votes24h = sp_db_one_int("SELECT COUNT(*) FROM {$T_V} WHERE created_at >= (NOW() - INTERVAL 1 DAY)", 0);
$votes7d  = sp_db_one_int("SELECT COUNT(*) FROM {$T_V} WHERE created_at >= (NOW() - INTERVAL 7 DAY)", 0);
$uniq24h  = sp_db_one_int("
  SELECT COUNT(DISTINCT IF(uid>0, CONCAT('U#',uid), CONCAT('G#',ip)))
  FROM {$T_V}
  WHERE created_at >= (NOW() - INTERVAL 1 DAY)
", 0);

// =====================================
//  2) Top sondages par nombre de votes
// =====================================
$topPolls = [];
$res = $xoopsDB->queryF("
  SELECT p.poll_id, p.question, COUNT(v.vote_id) AS vc
  FROM {$T_P} p
  LEFT JOIN {$T_V} v ON v.poll_id = p.poll_id
  GROUP BY p.poll_id, p.question
  ORDER BY vc DESC, p.poll_id DESC
  LIMIT 6
");
$topMax = 0;
if ($res) {
    while ($r = $xoopsDB->fetchArray($res)) {
        $id  = (int)$r['poll_id'];
        $q   = (string)$r['question'];
        $vc  = (int)$r['vc'];
        $topPolls[] = ['poll_id'=>$id,'question'=>$q,'votes'=>$vc];
        if ($vc > $topMax) $topMax = $vc;
    }
}

// =====================================
//  3) Derniers votes (10 plus récents)
// =====================================
$lastVotes = [];
$res = $xoopsDB->queryF("
  SELECT v.vote_id, v.poll_id, v.option_id, v.uid, v.ip, v.created_at,
         p.question, o.label AS option_label, u.uname
  FROM {$T_V} v
  LEFT JOIN {$T_P} p ON p.poll_id = v.poll_id
  LEFT JOIN {$T_O} o ON o.option_id = v.option_id
  LEFT JOIN {$T_U} u ON u.uid = v.uid
  ORDER BY v.created_at DESC, v.vote_id DESC
  LIMIT 10
");
if ($res) {
    while ($r = $xoopsDB->fetchArray($res)) {
        $lastVotes[] = [
            'vote_id'  => (int)$r['vote_id'],
            'poll_id'  => (int)$r['poll_id'],
            'question' => (string)($r['question'] ?? ''),
            'option'   => (string)($r['option_label'] ?? ('#'.(int)$r['option_id'].' (option supprimée)')),
            'uid'      => (int)$r['uid'],
            'uname'    => (string)($r['uname'] ?? ''),
            'ip'       => (string)$r['ip'],
            'dt'       => (string)$r['created_at'],
        ];
    }
}

// =====================================
//  4) Sondages qui se terminent bientôt
// =====================================
$closingSoon = [];
$res = $xoopsDB->queryF("
  SELECT poll_id, question, ends_at
  FROM {$T_P}
  WHERE status='published'
    AND ends_at IS NOT NULL
    AND ends_at <> '0000-00-00 00:00:00'
    AND ends_at >= NOW()
  ORDER BY ends_at ASC
  LIMIT 6
");
if ($res) {
    while ($r = $xoopsDB->fetchArray($res)) {
        $closingSoon[] = [
            'poll_id' => (int)$r['poll_id'],
            'question'=> (string)$r['question'],
            'ends_at' => (string)$r['ends_at'],
        ];
    }
}

// =====================================
//  5) Sondages ouverts sans vote
// =====================================
$openNoVotes = [];
$res = $xoopsDB->queryF("
  SELECT p.poll_id, p.question
  FROM {$T_P} p
  LEFT JOIN {$T_V} v ON v.poll_id = p.poll_id
  WHERE p.status='published'
    AND (p.starts_at IS NULL OR p.starts_at='' OR p.starts_at <= NOW())
    AND (p.ends_at  IS NULL OR p.ends_at=''  OR p.ends_at  = '0000-00-00 00:00:00' OR p.ends_at >= NOW())
  GROUP BY p.poll_id, p.question
  HAVING COUNT(v.vote_id) = 0
  ORDER BY p.poll_id DESC
  LIMIT 6
");
if ($res) {
    while ($r = $xoopsDB->fetchArray($res)) {
        $openNoVotes[] = [
            'poll_id' => (int)$r['poll_id'],
            'question'=> (string)$r['question'],
        ];
    }
}

// =====================================
//  6) Derniers sondages créés (6)
// =====================================
$recentPolls = [];
$res = $xoopsDB->queryF("
  SELECT poll_id, question, status, created_at
  FROM {$T_P}
  ORDER BY created_at DESC, poll_id DESC
  LIMIT 6
");
if ($res) {
    while ($r = $xoopsDB->fetchArray($res)) {
        $recentPolls[] = [
            'poll_id'   => (int)$r['poll_id'],
            'question'  => (string)$r['question'],
            'status'    => (string)$r['status'],
            'created_at'=> (string)$r['created_at'],
        ];
    }
}

// ===========================
//  Rendu Dashboard
// ===========================
echo '<style>
/* ===== Dashboard Sondages Pro ===== */
.sp-wrap{max-width:1200px;margin:0 auto;font:14px/1.45 system-ui, -apple-system, Segoe UI, Roboto, Ubuntu, Cantarell, Noto Sans, Arial, sans-serif}
.sp-grid{display:grid;grid-template-columns:repeat(12,1fr);gap:14px}
.sp-card{border:1px solid #e5e7eb;border-radius:12px;background:#fff;padding:14px}
.sp-kpis{grid-column:1/-1;display:grid;grid-template-columns:repeat(6,1fr);gap:12px}
.sp-kpi{border:1px solid #e5e7eb;border-radius:12px;background:#f8fafc;padding:12px}
.sp-kpi h4{margin:0 0 4px;font-size:.92rem;color:#334155}
.sp-kpi .num{font-size:1.6rem;font-weight:700}

.sp-col-6{grid-column:span 6}
.sp-col-4{grid-column:span 4}
.sp-col-8{grid-column:span 8}
.sp-col-12{grid-column:1/-1}

.sp-table{width:100%;border-collapse:separate;border-spacing:0;border:1px solid #e5e7eb;border-radius:12px;overflow:hidden}
.sp-table th{background:#f1f5f9;font-weight:600;text-align:left;padding:8px;border-bottom:1px solid #e5e7eb}
.sp-table td{padding:8px;border-bottom:1px solid #eef2f7}
.sp-table tr:last-child td{border-bottom:none}

.sp-actions{margin:10px 0 16px}
.sp-actions .button{display:inline-block;margin-right:8px;padding:6px 10px;border:1px solid #e5e7eb;border-radius:10px;background:#1370F2;color:#fff;text-decoration:none}
.sp-actions .button:hover{background:#0f5ed0}

.sp-sub{color:#64748b;font-size:.92rem;margin:4px 0 0}
.sp-badge{display:inline-block;padding:2px 8px;border-radius:999px;font-size:.8rem;border:1px solid #e5e7eb;background:#fff}
.sp-badge.ok{border-color:#bbf7d0;background:#f0fdf4;color:#14532d}
.sp-badge.warn{border-color:#fde68a;background:#fffbeb;color:#7c2d12}
.sp-badge.muted{background:#f8fafc;color:#475569}

.bar{height:8px;background:#e5e7eb;border-radius:999px;overflow:hidden}
.bar > span{display:block;height:100%;background:#1370F2}

@media (max-width:1024px){
  .sp-kpis{grid-template-columns:repeat(3,1fr)}
  .sp-col-6{grid-column:1/-1}
  .sp-col-4{grid-column:1/-1}
  .sp-col-8{grid-column:1/-1}
}
</style>';

echo '<div class="sp-wrap">';
echo '<h2>'._AM_SP_TITLE.' — Dashboard</h2>';
echo '<div class="sp-actions">';
echo '  <a class="button" href="poll.php?op=new">'._AM_SP_NEW.'</a>';
echo '  <a class="button" href="poll.php">'._AM_SP_POLLS.'</a>';
echo '  <a class="button" href="votes.php">'._AM_SP_VOTES.'</a>';
echo '</div>';

echo '<div class="sp-grid">';

// KPIs
echo '<div class="sp-kpis sp-card">';
echo '  <div class="sp-kpi"><h4>Total sondages</h4><div class="num">'.(int)$totalPolls.'</div></div>';
echo '  <div class="sp-kpi"><h4>Total options</h4><div class="num">'.(int)$totalOpts.'</div></div>';
echo '  <div class="sp-kpi"><h4>Total votes</h4><div class="num">'.(int)$totalVotes.'</div></div>';
echo '  <div class="sp-kpi"><h4>Ouverts maintenant</h4><div class="num">'.(int)$openNow.'</div></div>';
echo '  <div class="sp-kpi"><h4>Votes (24h)</h4><div class="num">'.(int)$votes24h.'</div><div class="sp-sub">'.(int)$uniq24h.' répondants uniques</div></div>';
echo '  <div class="sp-kpi"><h4>Votes (7 jours)</h4><div class="num">'.(int)$votes7d.'</div></div>';
echo '</div>';

// Répartition statuts + Top polls
echo '<div class="sp-col-6 sp-card">';
echo '  <h3>Répartition des statuts</h3>';
echo '  <table class="sp-table"><thead><tr><th>Statut</th><th>Nombre</th></tr></thead><tbody>';
echo '    <tr><td><span class="sp-badge ok">Publié</span></td><td>'.(int)$byStatus['published'].'</td></tr>';
echo '    <tr><td><span class="sp-badge warn">Brouillon</span></td><td>'.(int)$byStatus['draft'].'</td></tr>';
echo '    <tr><td><span class="sp-badge muted">Clôturé</span></td><td>'.(int)$byStatus['closed'].'</td></tr>';
echo '  </tbody></table>';
echo '</div>';

echo '<div class="sp-col-6 sp-card">';
echo '  <h3>Top sondages par nombre de votes</h3>';
if (!empty($topPolls)) {
    echo '<table class="sp-table"><thead><tr><th>#</th><th>Sondage</th><th style="width:45%">Volume</th><th>Votes</th></tr></thead><tbody>';
    foreach ($topPolls as $t) {
        $id = (int)$t['poll_id'];
        $q  = sp_h($t['question']);
        $vc = (int)$t['votes'];
        $pct = ($topMax > 0) ? max(4, (int)round($vc * 100 / $topMax)) : 0;
        $uView    = XOOPS_URL . '/modules/sondages_pro/view.php?poll_id='.$id;
        $uResults = XOOPS_URL . '/modules/sondages_pro/results.php?poll_id='.$id;
        echo '<tr>';
        echo '  <td>'.$id.'</td>';
        echo '  <td><a href="'.$uResults.'" target="_blank">'.$q.'</a><div class="sp-sub"><a href="'.$uView.'" target="_blank">Voir</a> · <a href="votes.php?poll_id='.$id.'">Votes</a></div></td>';
        echo '  <td><div class="bar"><span style="width:'.$pct.'%"></span></div></td>';
        echo '  <td>'.$vc.'</td>';
        echo '</tr>';
    }
    echo '</tbody></table>';
} else {
    echo '<div class="sp-sub">Aucun vote pour le moment.</div>';
}
echo '</div>';

// Derniers votes
echo '<div class="sp-col-8 sp-card">';
echo '  <h3>Derniers votes</h3>';
if (!empty($lastVotes)) {
    echo '<table class="sp-table"><thead><tr><th>ID</th><th>Sondage</th><th>Option</th><th>Utilisateur</th><th>IP</th><th>Date</th></tr></thead><tbody>';
    foreach ($lastVotes as $v) {
        $id  = (int)$v['vote_id'];
        $pid = (int)$v['poll_id'];
        $q   = sp_h($v['question']);
        $opt = sp_h($v['option']);
        $uid = (int)$v['uid'];
        $usr = $uid > 0 ? ('#'.$uid.' ('.sp_h($v['uname']).')') : 'Invité';
        $ip  = sp_h($v['ip']);
        $dt  = sp_h($v['dt']);
        $uRes = XOOPS_URL . '/modules/sondages_pro/results.php?poll_id='.$pid;
        echo '<tr>';
        echo '  <td>'.$id.'</td>';
        echo '  <td><a href="'.$uRes.'" target="_blank">'.$q.'</a></td>';
        echo '  <td>'.$opt.'</td>';
        echo '  <td>'.$usr.'</td>';
        echo '  <td>'.$ip.'</td>';
        echo '  <td>'.$dt.'</td>';
        echo '</tr>';
    }
    echo '</tbody></table>';
} else {
    echo '<div class="sp-sub">Aucun vote récent.</div>';
}
echo '</div>';

// Panneaux latéraux
echo '<div class="sp-col-4 sp-card">';
echo '  <h3>Se terminent bientôt</h3>';
if (!empty($closingSoon)) {
    echo '<table class="sp-table"><thead><tr><th>Sondage</th><th>Fin</th></tr></thead><tbody>';
    foreach ($closingSoon as $c) {
        $id = (int)$c['poll_id'];
        $q  = sp_h($c['question']);
        $ed = sp_h($c['ends_at']);
        $uRes = XOOPS_URL . '/modules/sondages_pro/results.php?poll_id='.$id;
        echo '<tr><td><a href="'.$uRes.'" target="_blank">'.$q.'</a></td><td>'.$ed.'</td></tr>';
    }
    echo '</tbody></table>';
} else {
    echo '<div class="sp-sub">Rien à venir sous 7–10 jours.</div>';
}

echo '  <h3 style="margin-top:14px">Ouverts sans vote</h3>';
if (!empty($openNoVotes)) {
    echo '<table class="sp-table"><thead><tr><th>Sondage</th><th>Actions</th></tr></thead><tbody>';
    foreach ($openNoVotes as $o) {
        $id = (int)$o['poll_id'];
        $q  = sp_h($o['question']);
        $uView = XOOPS_URL . '/modules/sondages_pro/view.php?poll_id='.$id;
        echo '<tr>';
        echo '  <td>'.$q.'</td>';
        echo '  <td><a href="poll.php?op=edit&poll_id='.$id.'">Éditer</a> · <a href="'.$uView.'" target="_blank">Voir</a></td>';
        echo '</tr>';
    }
    echo '</tbody></table>';
} else {
    echo '<div class="sp-sub">Tous les sondages ouverts ont au moins un vote.</div>';
}
echo '</div>';

// Derniers sondages créés
echo '<div class="sp-col-12 sp-card">';
echo '  <h3>Derniers sondages créés</h3>';
if (!empty($recentPolls)) {
    echo '<table class="sp-table"><thead><tr><th>#</th><th>Sondage</th><th>Statut</th><th>Créé le</th><th>Actions</th></tr></thead><tbody>';
    foreach ($recentPolls as $p) {
        $id = (int)$p['poll_id'];
        $q  = sp_h($p['question']);
        $st = (string)$p['status'];
        $dt = sp_h($p['created_at']);
        $stBadge = $st==='published' ? '<span class="sp-badge ok">Publié</span>' : ($st==='draft' ? '<span class="sp-badge warn">Brouillon</span>' : '<span class="sp-badge muted">Clôturé</span>');
        $uView    = XOOPS_URL . '/modules/sondages_pro/view.php?poll_id='.$id;
        $uResults = XOOPS_URL . '/modules/sondages_pro/results.php?poll_id='.$id;
        echo '<tr>';
        echo '  <td>'.$id.'</td>';
        echo '  <td>'.$q.'</td>';
        echo '  <td>'.$stBadge.'</td>';
        echo '  <td>'.$dt.'</td>';
        echo '  <td><a href="poll.php?op=edit&poll_id='.$id.'">Éditer</a> · <a href="'.$uView.'" target="_blank">Voir</a> · <a href="'.$uResults.'" target="_blank">Résultats</a> · <a href="votes.php?poll_id='.$id.'">Votes</a></td>';
        echo '</tr>';
    }
    echo '</tbody></table>';
} else {
    echo '<div class="sp-sub">Aucun sondage pour le moment.</div>';
}
echo '</div>';

echo '</div>'; // .sp-grid
echo '</div>'; // .sp-wrap

xoops_cp_footer();

