<?php
declare(strict_types=1);

require_once dirname(__DIR__, 3) . '/mainfile.php';
require_once XOOPS_ROOT_PATH . '/include/cp_header.php';

xoops_loadLanguage('admin', 'xoopspulse');

$xoopsUser   = $GLOBALS['xoopsUser'] ?? null;
$xoopsModule = $GLOBALS['xoopsModule'] ?? null;

if (!is_object($xoopsModule)) {
    $mh = xoops_getHandler('module');
    $xoopsModule = $mh ? $mh->getByDirname('xoopspulse') : null;
}

if (!is_object($xoopsUser) || !is_object($xoopsModule) || !$xoopsUser->isAdmin($xoopsModule->getVar('mid'))) {
    redirect_header(XOOPS_URL . '/', 3, defined('_AM_XOOPSPULSE_NOPERM') ? _AM_XOOPSPULSE_NOPERM : 'Accès refusé');
    exit;
}

$xoopsDB = $GLOBALS['xoopsDB'];
$table   = $xoopsDB->prefix('xoopspulse_hits');

/** ----- Param période ----- */
$days = isset($_GET['days']) ? (int)$_GET['days'] : 30;
$allowed = [0, 7, 30, 90];
if (!in_array($days, $allowed, true)) {
    $days = 30;
}

$conds = [];
if ($days > 0) {
    $since = time() - ($days * 86400);
    $conds[] = 'ts >= ' . (int)$since;
}
$where = $conds ? ('WHERE ' . implode(' AND ', $conds)) : '';

/** ----- Helpers DB ----- */
function topRows($db, string $sql): array {
    $out = [];
    $r = $db->query($sql);
    if (!$r) return $out;
    while ($row = $db->fetchArray($r)) $out[] = $row;
    return $out;
}
function q1($db, string $sql): int {
    $r = $db->query($sql);
    if (!$r) return 0;
    $row = $db->fetchRow($r);
    return $row ? (int)$row[0] : 0;
}
function wherePlus(string $where, string $cond): string {
    return $where === '' ? ('WHERE ' . $cond) : ($where . ' AND ' . $cond);
}

/** ----- Data sets ----- */
// Hits par jour (selon période)
$byDay = topRows($xoopsDB, "
    SELECT DATE(FROM_UNIXTIME(ts)) AS d, COUNT(*) AS c
    FROM $table
    $where
    GROUP BY d
    ORDER BY d ASC
");

// Hits par heure (48h fixes)
$byHour = topRows($xoopsDB, "
    SELECT DATE_FORMAT(FROM_UNIXTIME(ts),'%m-%d %H:00') AS h, COUNT(*) AS c
    FROM $table
    WHERE ts >= " . (int)(time() - 48 * 3600) . "
    GROUP BY h
    ORDER BY h ASC
");

// OS / navigateurs (selon période)
$os = topRows($xoopsDB, "
    SELECT os AS k, COUNT(*) AS c
    FROM $table
    $where
    GROUP BY os
    ORDER BY c DESC
    LIMIT 10
");
$br = topRows($xoopsDB, "
    SELECT browser AS k, COUNT(*) AS c
    FROM $table
    $where
    GROUP BY browser
    ORDER BY c DESC
    LIMIT 10
");

// Total / Mobile / Dark (selon période)
$total     = q1($xoopsDB, "SELECT COUNT(*) FROM $table $where");
$mobileYes = q1($xoopsDB, "SELECT COUNT(*) FROM $table " . wherePlus($where, "is_mobile=1"));
$mobileNo  = max(0, $total - $mobileYes);

$darkYes = q1($xoopsDB, "SELECT COUNT(*) FROM $table " . wherePlus($where, "darkmode=1"));
$darkNo  = max(0, $total - $darkYes);

// Top pages (selon période) — condition page<>'' gérée proprement
$pageConds = $conds;
$pageConds[] = "page <> ''";
$wherePages = 'WHERE ' . implode(' AND ', $pageConds);

$pages = topRows($xoopsDB, "
    SELECT page AS k, COUNT(*) AS c
    FROM $table
    $wherePages
    GROUP BY page
    ORDER BY c DESC
    LIMIT 12
");

/** ----- Convert helpers ----- */
$labelsByDay  = array_map(static fn($r) => (string)$r['d'], $byDay);
$dataByDay    = array_map(static fn($r) => (int)$r['c'], $byDay);

$labelsByHour = array_map(static fn($r) => (string)$r['h'], $byHour);
$dataByHour   = array_map(static fn($r) => (int)$r['c'], $byHour);

$labelsOs = array_map(static fn($r) => (string)($r['k'] ?? 'Other'), $os);
$dataOs   = array_map(static fn($r) => (int)$r['c'], $os);

$labelsBr = array_map(static fn($r) => (string)($r['k'] ?? 'Other'), $br);
$dataBr   = array_map(static fn($r) => (int)$r['c'], $br);

$labelsPages = array_map(static fn($r) => (string)($r['k'] ?? '(vide)'), $pages);
$dataPages   = array_map(static fn($r) => (int)$r['c'], $pages);

/** ----- Render ----- */
xoops_cp_header();

echo '<h2>XOOPS Pulse — Graphiques</h2>';

echo '<form method="get" style="margin:10px 0">';
echo '<label>Période : </label> ';
echo '<select name="days" onchange="this.form.submit()">';
foreach ([0=>'Tout', 7=>'7 jours', 30=>'30 jours', 90=>'90 jours'] as $k => $label) {
    $sel = ($days === (int)$k) ? ' selected' : '';
    echo '<option value="' . (int)$k . '"' . $sel . '>' . htmlspecialchars($label, ENT_QUOTES) . '</option>';
}
echo '</select>';
echo '</form>';

echo '<div class="outer" style="padding:10px;margin:10px 0">';
echo '<b>Total hits (période) :</b> ' . (int)$total . ' &nbsp; | &nbsp; ';
echo '<b>Mobile :</b> ' . (int)$mobileYes . ' &nbsp; | &nbsp; ';
echo '<b>Desktop :</b> ' . (int)$mobileNo . ' &nbsp; | &nbsp; ';
echo '<b>Dark mode :</b> ' . (int)$darkYes;
echo '</div>';

echo '
<style>
.xp-grid{display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:14px}
.xp-card{padding:10px; overflow:hidden}

.xp-chart{position:relative; height:320px;}
.xp-chart.square{height:320px; max-width:340px; margin:0 auto;}
.xp-chart.tall{height:520px;}

.xp-chart canvas{
  position:absolute; inset:0;
  width:100% !important;
  height:100% !important;
}

@media (max-width: 900px){
  .xp-grid{grid-template-columns:1fr}
  .xp-chart{height:280px;}
  .xp-chart.square{max-width:300px;}
  .xp-chart.tall{height:480px;}
}
</style>
';

echo '<div class="xp-grid">';
echo '<div class="outer xp-card"><h3>Hits par jour</h3><div class="xp-chart"><canvas id="c_byDay"></canvas></div></div>';
echo '<div class="outer xp-card"><h3>Activité (48h)</h3><div class="xp-chart"><canvas id="c_byHour"></canvas></div></div>';
echo '<div class="outer xp-card"><h3>Répartition OS</h3><div class="xp-chart square"><canvas id="c_os"></canvas></div></div>';
echo '<div class="outer xp-card"><h3>Répartition navigateurs</h3><div class="xp-chart square"><canvas id="c_br"></canvas></div></div>';
echo '<div class="outer xp-card"><h3>Mobile vs Desktop</h3><div class="xp-chart square"><canvas id="c_mobile"></canvas></div></div>';
echo '<div class="outer xp-card"><h3>Dark mode</h3><div class="xp-chart square"><canvas id="c_dark"></canvas></div></div>';
echo '<div class="outer xp-card" style="grid-column:1/-1"><h3>Top pages</h3><div class="xp-chart tall"><canvas id="c_pages"></canvas></div></div>';
echo '</div>';

/** ----- Données JS ----- */
$js = [
    'byDay'   => ['labels' => $labelsByDay,  'data' => $dataByDay],
    'byHour'  => ['labels' => $labelsByHour, 'data' => $dataByHour],
    'os'      => ['labels' => $labelsOs,     'data' => $dataOs],
    'br'      => ['labels' => $labelsBr,     'data' => $dataBr],
    'mobile'  => ['labels' => ['Mobile','Desktop'], 'data' => [(int)$mobileYes,(int)$mobileNo]],
    'dark'    => ['labels' => ['Dark','Light'],     'data' => [(int)$darkYes,(int)$darkNo]],
    'pages'   => ['labels' => $labelsPages,  'data' => $dataPages],
];

$chartPath = XOOPS_ROOT_PATH . '/modules/xoopspulse/assets/vendor/chartjs/chart.umd.min.js';
$chartVer  = is_file($chartPath) ? (string)filemtime($chartPath) : (string)time();

// évite mixed content
$baseUrl = XOOPS_URL;
$isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (isset($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443);
if ($isHttps) {
    $baseUrl = preg_replace('~^http://~i', 'https://', $baseUrl);
}

echo '<script src="' . $baseUrl . '/modules/xoopspulse/assets/vendor/chartjs/chart.umd.min.js?v=' . $chartVer . '"></script>';
echo '<script>window.__XP_CHARTS = ' . json_encode($js, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . ';</script>';


echo <<<'JS'
<script>
(function(){
  if(!window.Chart){
    console.error("XOOPS Pulse: Chart.js non chargé (404 ou script bloqué).");
    return;
  }

  const XP = window.__XP_CHARTS || {};
  const palette = [
    'rgba(54, 162, 235, 0.65)',
    'rgba(255, 99, 132, 0.65)',
    'rgba(255, 206, 86, 0.65)',
    'rgba(75, 192, 192, 0.65)',
    'rgba(153, 102, 255, 0.65)',
    'rgba(255, 159, 64, 0.65)',
    'rgba(201, 203, 207, 0.65)'
  ];
  function colors(n){
    n = Math.max(1, n|0);
    const out = [];
    for(let i=0;i<n;i++) out.push(palette[i % palette.length]);
    return out;
  }

  const common = {
    responsive: true,
    maintainAspectRatio: false
  };

  function makeLine(id, labels, data){
    const el = document.getElementById(id);
    if(!el) return;
    new Chart(el, {
      type: 'line',
      data: { labels: labels, datasets: [{ label: 'Hits', data: data, tension: 0.25, fill: true }] },
      options: Object.assign({}, common, {
        plugins: { legend: { display: false } },
        scales: { y: { beginAtZero: true } }
      })
    });
  }

  function makeDoughnut(id, labels, data){
    const el = document.getElementById(id);
    if(!el) return;
    new Chart(el, {
      type: 'doughnut',
      data: { labels: labels, datasets: [{ data: data, backgroundColor: colors(labels.length) }] },
      options: Object.assign({}, common, {
        plugins: { legend: { position: 'bottom' } }
      })
    });
  }

  function makeBar(id, labels, data, opts){
    const el = document.getElementById(id);
    if(!el) return;

    opts = opts || {};
    const horizontal = !!opts.horizontal;

    new Chart(el, {
      type: 'bar',
      data: {
        labels: labels,
        datasets: [{ label: 'Hits', data: data, backgroundColor: colors(labels.length) }]
      },
      options: Object.assign({}, common, {
        indexAxis: horizontal ? 'y' : 'x',
        plugins: {
          legend: { display: false },
          tooltip: {
            callbacks: {
              label: function(ctx){
                const v = horizontal ? ctx.parsed.x : ctx.parsed.y;
                return ' ' + v;
              }
            }
          }
        },
        scales: {
          x: { beginAtZero: true },
          y: { beginAtZero: true }
        }
      })
    });
  }

  makeLine('c_byDay',  (XP.byDay && XP.byDay.labels) || [],  (XP.byDay && XP.byDay.data) || []);
  makeLine('c_byHour', (XP.byHour && XP.byHour.labels) || [], (XP.byHour && XP.byHour.data) || []);

  makeDoughnut('c_os',     (XP.os && XP.os.labels) || [],     (XP.os && XP.os.data) || []);
  makeDoughnut('c_br',     (XP.br && XP.br.labels) || [],     (XP.br && XP.br.data) || []);
  makeDoughnut('c_mobile', (XP.mobile && XP.mobile.labels) || [], (XP.mobile && XP.mobile.data) || []);
  makeDoughnut('c_dark',   (XP.dark && XP.dark.labels) || [],   (XP.dark && XP.dark.data) || []);

  makeBar('c_pages', (XP.pages && XP.pages.labels) || [], (XP.pages && XP.pages.data) || [], { horizontal: true });
})();
</script>
JS;



xoops_cp_footer();

