<?php
declare(strict_types=1);


$GLOBALS['xoopsOption']['nocommon'] = 1;
require_once dirname(__DIR__, 2) . '/mainfile.php';

header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

function xp_int($v, int $min = 0, int $max = 30000): int {
    if ($v === null || $v === '') return 0;
    if (is_string($v)) $v = trim($v);
    if (!is_numeric($v)) return 0;
    $i = (int)$v;
    if ($i < $min) $i = $min;
    if ($i > $max) $i = $max;
    return $i;
}

function xp_str($v, int $maxLen = 255): string {
    if ($v === null) return '';
    $s = is_string($v) ? $v : (string)$v;
    $s = trim($s);
    if ($s === '') return '';
    if (strlen($s) > $maxLen) $s = substr($s, 0, $maxLen);
    return $s;
}

function anonymize_ip(string $ip): string {
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
        $p = explode('.', $ip);
        $p[3] = '0';
        return implode('.', $p);
    }
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
        $bin = @inet_pton($ip);
        if ($bin === false) return '';
        $masked = substr($bin, 0, 6) . str_repeat("\0", 10); // /48
        $out = @inet_ntop($masked);
        return is_string($out) ? $out : '';
    }
    return '';
}

function detect_os(string $ua): string {
    if (stripos($ua, 'Windows') !== false) return 'Windows';
    if (stripos($ua, 'Android') !== false) return 'Android';
    if (stripos($ua, 'iPhone') !== false || stripos($ua, 'iPad') !== false || stripos($ua, 'iOS') !== false) return 'iOS';
    if (stripos($ua, 'Macintosh') !== false || stripos($ua, 'Mac OS') !== false) return 'macOS';
    if (stripos($ua, 'Linux') !== false) return 'Linux';
    return 'Other';
}

function detect_browser(string $ua): string {
    if (stripos($ua, 'Edg/') !== false) return 'Edge';
    if (stripos($ua, 'OPR/') !== false) return 'Opera';
    if (stripos($ua, 'Firefox/') !== false) return 'Firefox';
    if (stripos($ua, 'Chrome/') !== false && stripos($ua, 'Edg/') === false) return 'Chrome';
    if (stripos($ua, 'Safari/') !== false && stripos($ua, 'Chrome/') === false) return 'Safari';
    return 'Other';
}

function b64url_decode(string $s): string {
    $s = strtr($s, '-_', '+/');
    $pad = strlen($s) % 4;
    if ($pad) $s .= str_repeat('=', 4 - $pad);
    $out = base64_decode($s, true);
    return is_string($out) ? $out : '';
}

function uid_from_xoops_jwt_cookie(): int {
    foreach ($_COOKIE as $name => $val) {
        if (strpos($name, 'xoops_user_') !== 0) continue;
        if (!is_string($val) || $val === '') continue;

        $token = urldecode($val);
        $parts = explode('.', $token);
        if (count($parts) < 2) continue;

        $payloadJson = b64url_decode($parts[1]);
        $payload = json_decode($payloadJson, true);

        if (is_array($payload)) {
            $uid = $payload['uid'] ?? $payload['user_id'] ?? $payload['sub'] ?? 0;
            $uid = (int)$uid;
            if ($uid > 0) return $uid;
        }
    }
    return 0;
}

function ip_in_cidr(string $ip, string $cidr): bool {
    if (strpos($cidr, '/') === false) return false;
    [$subnet, $bits] = explode('/', $cidr, 2);
    $bits = (int)$bits;

    $ipBin = @inet_pton($ip);
    $subBin = @inet_pton($subnet);
    if ($ipBin === false || $subBin === false) return false;

    $len = strlen($ipBin); // 4 or 16
    $maxBits = $len * 8;
    if ($bits < 0 || $bits > $maxBits) return false;

    $bytes = intdiv($bits, 8);
    $rem = $bits % 8;

    if ($bytes > 0 && substr($ipBin, 0, $bytes) !== substr($subBin, 0, $bytes)) return false;
    if ($rem === 0) return true;

    $mask = chr((0xFF << (8 - $rem)) & 0xFF);
    return ((ord($ipBin[$bytes]) & ord($mask)) === (ord($subBin[$bytes]) & ord($mask)));
}

function ip_in_list(string $ip, array $list): bool {
    foreach ($list as $entry) {
        if (!is_string($entry) || $entry === '') continue;
        if (strpos($entry, '/') !== false) {
            if (ip_in_cidr($ip, $entry)) return true;
        } else {
            if ($ip === $entry) return true;
        }
    }
    return false;
}

function respond_pixel(): void {
    header('Content-Type: image/gif');
    echo base64_decode('R0lGODlhAQABAPAAAP///wAAACH5BAAAAAAALAAAAAABAAEAAAICRAEAOw==');
    exit;
}

// --- config
$cfgFile = __DIR__ . '/include/pulse_config.php';
$cfg = is_file($cfgFile) ? require $cfgFile : [];
$cfg = is_array($cfg) ? $cfg : [];

$ignoreBots   = (bool)($cfg['ignore_bots'] ?? true);
$botRegex     = (string)($cfg['bot_regex'] ?? '~bot|crawl|spider~i');
$ignoreIps    = (array)($cfg['ignore_ips'] ?? []);
$allowIps     = (array)($cfg['allow_ips'] ?? []);
$ignoreUids   = (array)($cfg['ignore_uids'] ?? []);
$allowGuests  = (bool)($cfg['allow_guests'] ?? true);

// --- Lecture GET (pixel)
$req = $_GET;

$uid = xp_int($req['u'] ?? 0, 0, 999999999);
if ($uid === 0) {
    $uid = uid_from_xoops_jwt_cookie();
}

$ua = xp_str($_SERVER['HTTP_USER_AGENT'] ?? '', 2000);
$ip = (string)($_SERVER['REMOTE_ADDR'] ?? '');

// --- filtres allow/ignore
if ($ip !== '' && ip_in_list($ip, $allowIps)) {
    // force allow
} else {
    if ($ip !== '' && ip_in_list($ip, $ignoreIps)) {
        respond_pixel();
    }
    if (!$allowGuests && $uid === 0) {
        respond_pixel();
    }
    if (!empty($ignoreUids) && in_array($uid, $ignoreUids, true)) {
        respond_pixel();
    }
    if ($ignoreBots && $ua !== '' && preg_match($botRegex, $ua)) {
        // si l'utilisateur est loggé (uid>0), on conserve quand même
        if ($uid === 0) {
            respond_pixel();
        }
    }
}

// --- champs
$screenW = xp_int($req['sw'] ?? 0, 0, 30000);
$screenH = xp_int($req['sh'] ?? 0, 0, 30000);
$vpW     = xp_int($req['vw'] ?? 0, 0, 30000);
$vpH     = xp_int($req['vh'] ?? 0, 0, 30000);

$dpr100  = xp_int($req['dpr100'] ?? 0, 0, 1000);
$dpr     = $dpr100 > 0 ? number_format($dpr100 / 100, 2, '.', '') : '';

$lang    = xp_str($req['lg'] ?? '', 20);
$dark    = xp_int($req['dm'] ?? 0, 0, 1);

$tz = '';
if (!empty($req['tzb'])) {
    $decoded = b64url_decode(xp_str($req['tzb'], 200));
    $decoded = xp_str($decoded, 60);
    if ($decoded !== '') $tz = $decoded;
}

$isMobile = (int)preg_match('~Mobile|Android|iPhone|iPad~i', $ua);
$os = detect_os($ua);
$browser = detect_browser($ua);

$ipAnon = anonymize_ip($ip);

// Page + referrer depuis HTTP_REFERER
$referrer = '';
$page = '';
if (!empty($_SERVER['HTTP_REFERER'])) {
    $u = @parse_url((string)$_SERVER['HTTP_REFERER']);
    if (is_array($u)) {
        $referrer = xp_str($u['host'] ?? '', 255);
        $path = $u['path'] ?? '';
        $qry  = isset($u['query']) ? ('?' . $u['query']) : '';
        $page = xp_str($path . $qry, 255);
    }
}

$ts = time();

// --- Insert DB (mysqli direct)
$mysqli = @new mysqli(XOOPS_DB_HOST, XOOPS_DB_USER, XOOPS_DB_PASS, XOOPS_DB_NAME);
if (!$mysqli->connect_errno) {
    @$mysqli->set_charset('utf8mb4');

    $table = XOOPS_DB_PREFIX . '_xoopspulse_hits';

    $sql = "INSERT INTO `$table`
      (`ts`,`uid`,`ip_anon`,`ua`,`is_mobile`,`os`,`browser`,`screen_w`,`screen_h`,`vp_w`,`vp_h`,`dpr`,`lang`,`tz`,`darkmode`,`page`,`referrer`)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";

    $stmt = @$mysqli->prepare($sql);
    if ($stmt) {
        $types = "iississiiiisssiss";
        @$stmt->bind_param(
            $types,
            $ts,
            $uid,
            $ipAnon,
            $ua,
            $isMobile,
            $os,
            $browser,
            $screenW,
            $screenH,
            $vpW,
            $vpH,
            $dpr,
            $lang,
            $tz,
            $dark,
            $page,
            $referrer
        );
        @$stmt->execute();
        @$stmt->close();
    }

    // --- Purge auto 1x/jour : supprime > 90 jours
    $purgeDays = 90;
    $purgeFile = XOOPS_ROOT_PATH . '/cache/xoopspulse_last_purge.txt';
    $lockFile  = XOOPS_ROOT_PATH . '/cache/xoopspulse_purge.lock';
    $now = time();

    $lock = @fopen($lockFile, 'c');
    if ($lock) {
        if (@flock($lock, LOCK_EX | LOCK_NB)) {
            $last = (int)@file_get_contents($purgeFile);
            if ($last <= 0 || ($now - $last) > 86400) {
                $threshold = $now - ($purgeDays * 86400);
                @ $mysqli->query("DELETE FROM `$table` WHERE `ts` < " . (int)$threshold);
                @file_put_contents($purgeFile, (string)$now, LOCK_EX);
            }
            @flock($lock, LOCK_UN);
        }
        @fclose($lock);
    }

    @$mysqli->close();
}

respond_pixel();

